// Protocol Buffers - Google's data interchange format
// Copyright 2008 Google Inc.  All rights reserved.
//
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file or at
// https://developers.google.com/open-source/licenses/bsd

// Author: kenton@google.com (Kenton Varda)
//  Based on original Protocol Buffers design by
//  Sanjay Ghemawat, Jeff Dean, and others.

#ifndef GOOGLE_PROTOBUF_COMPILER_JAVA_MESSAGE_H__
#define GOOGLE_PROTOBUF_COMPILER_JAVA_MESSAGE_H__

#include <string>

#include "absl/container/btree_map.h"
#include "google/protobuf/compiler/java/field.h"

namespace google {
namespace protobuf {
namespace compiler {
namespace java {
class Context;            // context.h
class ClassNameResolver;  // name_resolver.h
}  // namespace java
}  // namespace compiler
namespace io {
class Printer;  // printer.h
}
}  // namespace protobuf
}  // namespace google

namespace google {
namespace protobuf {
namespace compiler {
namespace java {

static const int kMaxStaticSize = 1 << 15;  // aka 32k

class MessageGenerator {
 public:
  explicit MessageGenerator(const Descriptor* descriptor);
  MessageGenerator(const MessageGenerator&) = delete;
  MessageGenerator& operator=(const MessageGenerator&) = delete;
  virtual ~MessageGenerator();

  // All static variables have to be declared at the top-level of the file
  // so that we can control initialization order, which is important for
  // DescriptorProto bootstrapping to work.
  virtual void GenerateStaticVariables(io::Printer* printer,
                                       int* bytecode_estimate) = 0;

  // Output code which initializes the static variables generated by
  // GenerateStaticVariables(). Returns an estimate of bytecode size.
  virtual int GenerateStaticVariableInitializers(io::Printer* printer) = 0;

  // Generate the class itself.
  virtual void Generate(io::Printer* printer) = 0;

  // Generates the base interface that both the class and its builder
  // implement
  virtual void GenerateInterface(io::Printer* printer) = 0;

  // Generate code to register all contained extensions with an
  // ExtensionRegistry.
  virtual void GenerateExtensionRegistrationCode(io::Printer* printer) = 0;
  virtual void GenerateKotlinDsl(io::Printer* printer) const = 0;
  virtual void GenerateKotlinMembers(io::Printer* printer) const = 0;
  virtual void GenerateTopLevelKotlinMembers(io::Printer* printer) const = 0;

 protected:
  const Descriptor* descriptor_;
  absl::btree_map<int, const OneofDescriptor*> oneofs_;
};

class ImmutableMessageGenerator : public MessageGenerator {
 public:
  ImmutableMessageGenerator(const Descriptor* descriptor, Context* context);
  ImmutableMessageGenerator(const ImmutableMessageGenerator&) = delete;
  ImmutableMessageGenerator& operator=(const ImmutableMessageGenerator&) =
      delete;
  ~ImmutableMessageGenerator() override;

  void Generate(io::Printer* printer) override;
  void GenerateInterface(io::Printer* printer) override;
  void GenerateExtensionRegistrationCode(io::Printer* printer) override;
  void GenerateStaticVariables(io::Printer* printer,
                               int* bytecode_estimate) override;

  // Returns an estimate of the number of bytes the printed code will compile to
  int GenerateStaticVariableInitializers(io::Printer* printer) override;
  void GenerateKotlinDsl(io::Printer* printer) const override;
  void GenerateKotlinMembers(io::Printer* printer) const override;
  void GenerateTopLevelKotlinMembers(io::Printer* printer) const override;

 private:
  void GenerateFieldAccessorTable(io::Printer* printer, int* bytecode_estimate);

  // Returns an estimate of the number of bytes the printed code will compile to
  int GenerateFieldAccessorTableInitializer(io::Printer* printer);

  void GenerateMessageSerializationMethods(io::Printer* printer);
  void GenerateParseFromMethods(io::Printer* printer);

  void GenerateBuilder(io::Printer* printer);
  void GenerateIsInitialized(io::Printer* printer);
  void GenerateDescriptorMethods(io::Printer* printer);
  void GenerateInitializers(io::Printer* printer);
  void GenerateEqualsAndHashCode(io::Printer* printer);
  void GenerateParser(io::Printer* printer);
  void GenerateParsingConstructor(io::Printer* printer);
  void GenerateMutableCopy(io::Printer* printer);
  void GenerateKotlinExtensions(io::Printer* printer) const;
  void GenerateKotlinOrNull(io::Printer* printer) const;
  void GenerateAnyMethods(io::Printer* printer);

  Context* context_;
  ClassNameResolver* name_resolver_;
  FieldGeneratorMap<ImmutableFieldGenerator> field_generators_;
};

}  // namespace java
}  // namespace compiler
}  // namespace protobuf
}  // namespace google

#endif  // GOOGLE_PROTOBUF_COMPILER_JAVA_MESSAGE_H__
